const mongoose = require("mongoose");

const personalIdSchema = new mongoose.Schema({
  type: { type: String, enum: ["national_id", "passport", "other"], default: "national_id" },
  number: String,
  issuedBy: String,
  issueDate: String,
}, { _id: false });

const medicalRecordSchema = new mongoose.Schema({
  hospitalId: String,
  hospitalName: String,
  recordId: String,
  lastVisit: String,
  department: String,
}, { _id: false });

const insuranceSchema = new mongoose.Schema({
  provider: String,
  policyNumber: String,
  coverageStart: String,
  coverageEnd: String,
}, { _id: false });

const emergencyContactSchema = new mongoose.Schema({
  name: String,
  relation: String,
  phone: String,
  email: String,
}, { _id: false });

const memberSchema = new mongoose.Schema({
  memberId: String,
  relation: String,
  firstName: String,
  lastName: String,
  personalId: {
    type: { type: String, enum: ["national_id", "passport", "other"], default: "national_id" },
    number: String,
  },
}, { _id: false });

const addressSchema = new mongoose.Schema({
  detail: String, // Địa chỉ chi tiết (số nhà, tên đường)
  province: String, // Tỉnh/Thành phố
  district: String, // Quận/Huyện
  ward: String, // Phường/Xã
  postalCode: String, // Mã bưu chính
}, { _id: false });

const personalIdVerificationSchema = new mongoose.Schema({
  status: {
    type: String,
    enum: ["pending", "verified", "rejected"],
    default: "pending"
  },
  frontImage: String, // URL hoặc path to front image
  backImage: String, // URL hoặc path to back image
  verifiedAt: Date,
  verifiedBy: String, // Admin ID who verified
  rejectionReason: String,
  submittedAt: { type: Date, default: Date.now },
}, { _id: false });

const personalInfoSchema = new mongoose.Schema({
  firstName: String,
  lastName: String,
  birthday: String,
  address: String, // Địa chỉ cũ (backward compatible)
  addressDetail: addressSchema, // Địa chỉ mới với địa giới hành chính
  personalId: personalIdSchema,
  personalIdVerification: personalIdVerificationSchema,
  gender: { type: String, enum: ["male", "female", "other"] },
}, { _id: false });

const userSchema = new mongoose.Schema({
  email: { 
    type: String, 
    unique: true,
    sparse: true,
    lowercase: true,
    trim: true
  },
  phone: { 
    type: String, 
    trim: true
  },
  password: { 
    type: String, 
    required: true 
  },
  personalInfo: personalInfoSchema,
  medicalRecords: [medicalRecordSchema],
  insurance: insuranceSchema,
  emergencyContact: emergencyContactSchema,
  members: [memberSchema],
  permissions: { type: [String], default: [] },
  status: { 
    type: String, 
    enum: ["active", "inactive", "suspended"], 
    default: "active" 
  },
  verified: { type: Boolean, default: false },
  refreshTokens: [{
    token: String,
    createdAt: { type: Date, default: Date.now, expires: 2592000 } // 30 days
  }],
}, {
  timestamps: true,
  collection: 'users'
});

// Index for phone if provided
userSchema.index({ phone: 1 }, { sparse: true, unique: true });

module.exports = mongoose.model("User", userSchema);
