const User = require("../models/User");
const path = require("path");
const fs = require("fs");

// Helper to handle errors consistently
const handleError = (res, error, defaultMessage = "Lỗi server") => {
  console.error('Error:', error);
  
  if (error.code === 11000) {
    // Duplicate key error
    const field = Object.keys(error.keyPattern)[0];
    return res.status(400).json({ 
      message: field === 'email' ? "Email đã được sử dụng" : "Số điện thoại đã được sử dụng" 
    });
  }
  
  if (error.name === 'ValidationError') {
    return res.status(400).json({ 
      message: Object.values(error.errors).map(e => e.message).join(', ')
    });
  }
  
  return res.status(500).json({ 
    message: error.message || defaultMessage 
  });
};

// Helper function to ensure uploads directory exists
const ensureUploadsDir = () => {
  const uploadsDir = path.join(__dirname, '../uploads/cccd');
  if (!fs.existsSync(uploadsDir)) {
    fs.mkdirSync(uploadsDir, { recursive: true });
  }
  return uploadsDir;
};

exports.getProfile = async (req, res) => {
  try {
    const user = await User.findById(req.user._id).select("-password -refreshTokens");
    
    if (!user) {
      return res.status(404).json({ message: "Người dùng không tồn tại" });
    }

    res.json({ user });
  } catch (error) {
    return handleError(res, error, "Lỗi khi lấy thông tin người dùng");
  }
};

// Cập nhật thông tin cá nhân
exports.updatePersonalInfo = async (req, res) => {
  try {
    const userId = req.user._id;
    const { firstName, lastName, birthday, gender, personalId } = req.body;

    const updateData = {};
    if (firstName !== undefined) updateData["personalInfo.firstName"] = firstName;
    if (lastName !== undefined) updateData["personalInfo.lastName"] = lastName;
    if (birthday !== undefined) updateData["personalInfo.birthday"] = birthday;
    if (gender !== undefined) updateData["personalInfo.gender"] = gender;
    
    if (personalId) {
      if (personalId.type) updateData["personalInfo.personalId.type"] = personalId.type;
      if (personalId.number) updateData["personalInfo.personalId.number"] = personalId.number;
      if (personalId.issuedBy) updateData["personalInfo.personalId.issuedBy"] = personalId.issuedBy;
      if (personalId.issueDate) updateData["personalInfo.personalId.issueDate"] = personalId.issueDate;
    }

    const updatedUser = await User.findByIdAndUpdate(
      userId,
      { $set: updateData },
      { new: true, runValidators: true }
    ).select("-password -refreshTokens");

    if (!updatedUser) {
      return res.status(404).json({ message: "Người dùng không tồn tại" });
    }

    res.json({ 
      message: "Cập nhật thông tin cá nhân thành công",
      user: updatedUser 
    });
  } catch (error) {
    return handleError(res, error, "Lỗi khi cập nhật thông tin");
  }
};

// Cập nhật địa chỉ với địa giới hành chính
exports.updateAddress = async (req, res) => {
  try {
    const userId = req.user._id;
    const { detail, province, district, ward, postalCode } = req.body;

    const updateData = {};
    if (detail !== undefined) updateData["personalInfo.addressDetail.detail"] = detail;
    if (province !== undefined) updateData["personalInfo.addressDetail.province"] = province;
    if (district !== undefined) updateData["personalInfo.addressDetail.district"] = district;
    if (ward !== undefined) updateData["personalInfo.addressDetail.ward"] = ward;
    if (postalCode !== undefined) updateData["personalInfo.addressDetail.postalCode"] = postalCode;

    // Also update old address field for backward compatibility
    if (detail || province || district || ward) {
      const addressParts = [detail, ward, district, province].filter(Boolean);
      updateData["personalInfo.address"] = addressParts.join(", ");
    }

    const updatedUser = await User.findByIdAndUpdate(
      userId,
      { $set: updateData },
      { new: true, runValidators: true }
    ).select("-password -refreshTokens");

    if (!updatedUser) {
      return res.status(404).json({ message: "Người dùng không tồn tại" });
    }

    res.json({ 
      message: "Cập nhật địa chỉ thành công",
      user: updatedUser 
    });
  } catch (error) {
    return handleError(res, error, "Lỗi khi cập nhật thông tin");
  }
};

// Cập nhật thông tin bảo hiểm
exports.updateInsurance = async (req, res) => {
  try {
    const userId = req.user._id;
    const { provider, policyNumber, coverageStart, coverageEnd } = req.body;

    const updateData = {};
    if (provider !== undefined) updateData["insurance.provider"] = provider;
    if (policyNumber !== undefined) updateData["insurance.policyNumber"] = policyNumber;
    if (coverageStart !== undefined) updateData["insurance.coverageStart"] = coverageStart;
    if (coverageEnd !== undefined) updateData["insurance.coverageEnd"] = coverageEnd;

    const updatedUser = await User.findByIdAndUpdate(
      userId,
      { $set: updateData },
      { new: true, runValidators: true }
    ).select("-password -refreshTokens");

    if (!updatedUser) {
      return res.status(404).json({ message: "Người dùng không tồn tại" });
    }

    res.json({ 
      message: "Cập nhật thông tin bảo hiểm thành công",
      user: updatedUser 
    });
  } catch (error) {
    return handleError(res, error, "Lỗi khi cập nhật thông tin");
  }
};

// Cập nhật liên hệ khẩn cấp
exports.updateEmergencyContact = async (req, res) => {
  try {
    const userId = req.user._id;
    const { name, relation, phone, email } = req.body;

    const updateData = {};
    if (name !== undefined) updateData["emergencyContact.name"] = name;
    if (relation !== undefined) updateData["emergencyContact.relation"] = relation;
    if (phone !== undefined) updateData["emergencyContact.phone"] = phone;
    if (email !== undefined) updateData["emergencyContact.email"] = email;

    const updatedUser = await User.findByIdAndUpdate(
      userId,
      { $set: updateData },
      { new: true, runValidators: true }
    ).select("-password -refreshTokens");

    if (!updatedUser) {
      return res.status(404).json({ message: "Người dùng không tồn tại" });
    }

    res.json({ 
      message: "Cập nhật liên hệ khẩn cấp thành công",
      user: updatedUser 
    });
  } catch (error) {
    return handleError(res, error, "Lỗi khi cập nhật thông tin");
  }
};

// Xác minh CCCD - Upload ảnh
exports.verifyCCCD = async (req, res) => {
  try {
    const userId = req.user._id;
    const user = await User.findById(userId);

    if (!user) {
      return res.status(404).json({ message: "Người dùng không tồn tại" });
    }

    // Check if files were uploaded
    if (!req.files || (!req.files.frontImage && !req.files.backImage)) {
      return res.status(400).json({ message: "Vui lòng upload ảnh mặt trước và mặt sau của CCCD" });
    }

    const uploadsDir = ensureUploadsDir();
    const updateData = {
      "personalInfo.personalIdVerification.status": "pending",
      "personalInfo.personalIdVerification.submittedAt": new Date(),
    };

    // Save front image
    if (req.files.frontImage) {
      const frontFile = Array.isArray(req.files.frontImage) ? req.files.frontImage[0] : req.files.frontImage;
      const frontFileName = `${userId}_front_${Date.now()}${path.extname(frontFile.originalname)}`;
      const frontFilePath = path.join(uploadsDir, frontFileName);
      fs.writeFileSync(frontFilePath, frontFile.buffer);
      updateData["personalInfo.personalIdVerification.frontImage"] = `/uploads/cccd/${frontFileName}`;
    }

    // Save back image
    if (req.files.backImage) {
      const backFile = Array.isArray(req.files.backImage) ? req.files.backImage[0] : req.files.backImage;
      const backFileName = `${userId}_back_${Date.now()}${path.extname(backFile.originalname)}`;
      const backFilePath = path.join(uploadsDir, backFileName);
      fs.writeFileSync(backFilePath, backFile.buffer);
      updateData["personalInfo.personalIdVerification.backImage"] = `/uploads/cccd/${backFileName}`;
    }

    const updatedUser = await User.findByIdAndUpdate(
      userId,
      { $set: updateData },
      { new: true, runValidators: true }
    ).select("-password -refreshTokens");

    res.json({ 
      message: "Đã gửi yêu cầu xác minh CCCD. Vui lòng chờ xử lý.",
      user: updatedUser 
    });
  } catch (error) {
    return handleError(res, error, "Lỗi khi cập nhật thông tin");
  }
};

// Legacy update method - kept for backward compatibility
exports.updateProfile = async (req, res) => {
  try {
    const userId = req.user._id;
    const updateData = req.body;

    // Don't allow password update through this endpoint
    delete updateData.password;
    delete updateData._id;
    delete updateData.createdAt;

    // Handle email uniqueness check
    if (updateData.email) {
      const existingUser = await User.findOne({ 
        email: updateData.email.toLowerCase().trim(),
        _id: { $ne: userId }
      });
      if (existingUser) {
        return res.status(400).json({ message: "Email đã được sử dụng" });
      }
      updateData.email = updateData.email.toLowerCase().trim();
    }

    // Handle phone uniqueness check
    if (updateData.phone) {
      const existingUser = await User.findOne({ 
        phone: updateData.phone.trim(),
        _id: { $ne: userId }
      });
      if (existingUser) {
        return res.status(400).json({ message: "Số điện thoại đã được sử dụng" });
      }
      updateData.phone = updateData.phone.trim();
    }

    // Update user
    const updatedUser = await User.findByIdAndUpdate(
      userId,
      { $set: updateData },
      { new: true, runValidators: true }
    ).select("-password -refreshTokens");

    if (!updatedUser) {
      return res.status(404).json({ message: "Người dùng không tồn tại" });
    }

    res.json({ 
      message: "Cập nhật thông tin thành công",
      user: updatedUser 
    });
  } catch (error) {
    if (error.code === 11000) {
      const field = Object.keys(error.keyPattern)[0];
      return res.status(400).json({ 
        message: field === 'email' ? "Email đã được sử dụng" : "Số điện thoại đã được sử dụng" 
      });
    }
    return handleError(res, error, "Lỗi khi cập nhật thông tin");
  }
};
